//
//  BLEReader.h
//  BLEReader
//
//  Created by Innosis on 2024.
//  Copyright (c) 2024 Innosis. All rights reserved.
//

#import <Foundation/Foundation.h>

//General Error Code
#define BT_OK                       0
#define BTERR                       -100
#define BTERR_INVALID_PARAMETER     -101
#define BTERR_NO_DEV                -102
#define BTERR_OPEN_DEV_FAIL         -103
#define BTERR_DISABLED              -104
#define BTERR_NOT_CONNECTED         -105
#define BTERR_NO_IO_STREAM          -106
#define BTERR_TIMEOUT               -107
#define BTERR_IO                    -108
#define BTERR_NO_RESP               -109
#define BTERR_FRAME                 -110
#define BTERR_BATT_LOW              -111
#define BTERR_SAM_AUTH              -112

//Biometric Error Code
//No error
#define BIO_OK                      0
//Biometrics device performed an internal error
#define BIOERR_INTERNAL             -1
//Communication protocol error
#define BIOERR_PROTOCOLE            -2
//Can not connect biometrics device
#define BIOERR_CONNECT              -3
//Not enough memory for the creation of a database in module
#define BIOERR_NO_SPACE_LEFT        -7 
//unknown status error
#define BIOERR_STATUS               -9
//The database is full
#define BIOERR_DB_FULL              -10
//The database is empty
#define BIOERR_DB_EMPTY             -11
//User has already been enrolled
#define BIOERR_ALREADY_ENROLLED     -12
//The specified base does not exist
#define BIOERR_BASE_NOT_FOUND       -13
//The specified base already exist
#define BIOERR_BASE_ALREADY_EXISTS  -14
//The template is not valid
#define BIOERR_INVALID_TEMPLATE     -17
//No response after defined time
#define BIOERR_TIMEOUT             -19
//Invalid PK format
#define BIOERR_INVALID_PK_FORMAT     -27
//The user data are not valid
#define BIOERR_INVALID_USER_ID      -30
//User is not found
#define BIOERR_USER_NOT_FOUND       -33

#define BIOERR_DEV_NOT_CONNECTED    -40

//smartcard slot
#define SC_CARD_SLOT                0x00
#define SC_SAM_SLOT                 0x02

//Smartcard Error Code
#define SC_CARD_PRESENT             1
#define SC_CARD_ABSENT              0
#define SCERR                       -150
#define SCERR_INVALID_PARAMETER     -151
#define SCERR_READER_NOT_AVAILABLE  -152
#define SCERR_CARD_NO_RSP           -153
#define SCERR_OPEN_READER_FAIL      -154
#define SCERR_NO_DEV                -155
#define SCERR_CARD_REMOVED          -156
#define SCERR_NO_CARD               -157
#define SCERR_DEV_NOT_CONNECT       -158
#define SCERR_READER_UNAVAILABLE    -159
#define SCERR_SHARING_VIOLATION     -160
#define SCERR_TIMEOUT               -161
#define SCERR_UNRESPONSIVE_CARD     -162
#define SCERR_UNSUPPORTED_CARD      -163
#define SCERR_UNPOWERED_CARD        -164
#define SCERR_RESET_CARD            -165
#define SCERR_INVALID_HANDLE        -166
#define SCERR_AUTHENTICATION        -167


// Biometric template format type
// For Bio_EnrollExportPK and Bio_VerifyPK
#define PK_COMP_V2                  0x00
#define PK_MAT                      0x02
#define PK_MAT_NORM                 0x03 //for verify only
#define ANSI_INCITS_378_2004        0x41
#define ISO_19794_2_FMR_2011        0x4C //ISO/IEC 19794-2 Finger Minutiae Record version 2011
#define ANSI_INCITS_378_2009        0x4D
#define ISO_19794_2_FMR_CS          0x6C //ISO/IEC 19794-2 Finger Minutiae Card Record, Compact Size.
#define ISO_19794_2_FMR_NS          0x6D //ISO/IEC 19794-2 Finger Minutiae Card Record, Normal Size.
#define ISO_19794_2_FMR             0x6E
#define MINEX_A                     0x6F
#define DIN_V66400_FMR              0x7D
#define DIN_V66400_FMR_CS_AA        0x7E
#define ISO_19794_2_FMR_CS_AA       0x7F //ISO/IEC 19794-2 Finger Minutiae Card Record, Compact Size, minutiae ordered by Ascending Angle

//Enrollment / verification status code
#define BIO_NO_FINGER               0x00
#define BIO_FINGER_UP               0x01
#define BIO_FINGER_DOWN             0x02
#define BIO_FINGER_LEFT             0x03
#define BIO_FINGER_RIGHT            0x04
#define BIO_FINGER_PRESS_HARDER     0x05
#define BIO_FINGER_LATENT           0x06
#define BIO_FINGER_REMOVE           0x07
#define BIO_FINGER_OK               0x08
#define BIO_FINGER_DETECTED         0x09
#define BIO_FINGER_MATCHED          0
#define BIO_FINGER_NOT_MATCH        -8

#define BIO_FINGER_MATCHED          0
#define BIO_FINGER_NOT_MATCH        -8

//BT status
#define BLE_TIMEOUT                 2
#define BLE_CONNECTED               1
#define BLE_NOT_CONNECTED           0

#define BUFFER_SIZE 4000+15

@protocol BLEReaderDelegate;

@interface BLEReader : NSObject {
@private
    BOOL _waitFlag;
    BOOL _waitAsyncFlag;
    BOOL _waitMyKadImageFlag;
    BOOL _waitBeaconFlag;
    //error
    NSInteger _error;
    NSString* _errorDesc;
    
    int _packetType;
    int _expectedLen;
    uint8_t _buffer[BUFFER_SIZE];
    int _bufferLen;
}

@property(assign) id <BLEReaderDelegate> delegate;



// MyKad data field
struct CardInfo {
    uint8_t bOriginalName[150];
    uint8_t bGMPCName[80];
    uint8_t bKPTShortName[40];
    uint8_t bID[13];
    uint8_t bGender[1];
    uint8_t bOldID[8];
    uint8_t bBirthDate[4];
    uint8_t bBirthPlace[25];
    uint8_t bIssueDate[4];
    uint8_t bCitizenship[18];
    uint8_t bRace[25];
    uint8_t bReligion[11];
    uint8_t bEastMalaysian[1];
    uint8_t bRJ[2];
    uint8_t bKT[2];
    uint8_t bOtherID[11];
    uint8_t bCategory[1];
    uint8_t bCardVer[1];
    uint8_t bGreenCardExp[4];
    uint8_t bGreenCardNationality[20];
    
    uint8_t bAddress1[30];
    uint8_t bAddress2[30];
    uint8_t bAddress3[30];
    uint8_t bPostcode[3];
    uint8_t bCity[25];
    uint8_t bState[30];
};

struct MyKidInfo {
    //MyKid
    //Child info, JPN #1
    unsigned char uacJPN1Version[1];
    unsigned char uacJPN1Size[2];
    unsigned char uacBirthCertNo[15];
    unsigned char uacIDNum[12];
    unsigned char uacName[150];
    unsigned char uacGender[1];
    unsigned char uacCitizenship[17];
    unsigned char uacBirthState[30]; //state where child were born
    unsigned char uacBirthAddr1[30];
    unsigned char uacBirthAddr2[30];
    unsigned char uacBirthAddr3[30];
    unsigned char uacBirthPostcode[3];
    unsigned char uacBirthCity[30];
    unsigned char uacBirthAddState[30]; //state according to address
    unsigned char uacFReligion[14];
    unsigned char uacMReligion[14];
    
    //Child info, JPN #2 (for mykid issue from beginning 2003 to April 2003
    unsigned char uacJPN2Version[1]; //version: _1
    unsigned char uacJPN2Size[2];
    unsigned char uacBirthDate[4];
    unsigned char uacBirthTime[3];
    unsigned char uacBirthLocation1[40];
    unsigned char uacBirthLocation2[40];
    unsigned char uacRegistrationDate[4];
    unsigned char uacNewName[150];
    unsigned char uacNewAddr1[30];
    unsigned char uacNewAddr2[30];
    unsigned char uacNewAddr3[30];
    unsigned char uacNewPostcode[3];
    unsigned char uacNewCity[30];
    unsigned char uacNewState[30];
    
    //Mother info
    unsigned char uacMIDNum[12];
    unsigned char uacMOtherDocNum[15];
    unsigned char uacMOtherDocType[50];
    unsigned char uacMName[150];
    unsigned char uacMBirthDate[4];
    unsigned char uacMCitizenship[23];
    unsigned char uacMRace[30];
    
    //Father info
    unsigned char uacFIDNum[12];
    unsigned char uacFOtherDocNum[15];
    unsigned char uacFOtherDocType[50];
    unsigned char uacFName[150];
    unsigned char uacFBirthDate[4];
    unsigned char uacFCitizenship[23];
    unsigned char uacFRace[30];
    
    unsigned char uacRegistrationPlace[50];
    
    
    //Child info, JPN #2 (for mykid issue from beginning May 2003 to April 2003 "version: _2"
    unsigned char uacMAge[4]; //only in Version:_2, uacMBirthDate[4] is in Version:_1
    unsigned char uacFAge[4]; //only in Version:_2, uacFBirthDate[4] is in Version:_1
    
    //Contact person, JPN #4
    unsigned char bMyKidJPN4Ver[1];
    unsigned char bMyKidJPN4Size[2];
    unsigned char bMyKidContactName[25];
    unsigned char bMyKidContactAddress[50];
    unsigned char bMyKidContactTelNum[7];
};


//pinpad
- (int) GetUserPin:(id)viewController :(Boolean)mode :(NSString *)sCardPan :(NSString *)sSessionKey;
- (int) GetUserPinDirect:(id)viewController :(NSString *)sCardPan;
- (int) GetPSE:(uint8_t *)oPANBuf :(uint8_t *)oTrack2ascii;

// Device functions
- (id) init;                            //search and connect to BT device
- (void) disconnectBTDevice;            //disconnect BT device
- (void) BLEStartScan:(int)iTimeout;                  //Start BLE Scan
- (NSString *) GetDeviceUUID;           //Return BLE device UUID string
//- (void) ConnectUUID:(NSString *)sUUID; //Connect to BLE device with UUID
- (void) ConnectUUID:(NSString *)sUUID :(int)iTimeout;
- (int) IsDeviceConnect;                //check BT device connection status :: return 1=device connected,                                      //0=device not ready
- (NSString*) GetSDKVersion;
- (int) GetBatteryLevel;
- (unsigned int) Buzzer_Beep;
- (unsigned int) GetTID:(uint8_t *)tidData;
- (unsigned int) GetFWVersion:(uint8_t *)fwData :(int *)fwLen;

// Biometric function
- (unsigned int) btConnectBio;          //Connect to biometric module
- (unsigned int) btDisconnectBio;       //disconnect biometric module
- (unsigned int) btBioVerifyRight:(int)iTimeout :(int **)iMatchStatus;  //Verify with MyKad Right minutiae
- (unsigned int) btBioVerifyLeft:(int)iTimeout :(int **)iMatchStatus;   //verify with MyKad Left minutiae
- (unsigned int) btBioVerifyBothFP:(int)iTimeout :(int **)iMatchStatus; //verify against 2 templates
- (unsigned int) btBioVerifyBothFPNoAsync:(int)iTimeout :(int **)iMatchStatus;
- (unsigned int) btBioVerifyRightNoAsync:(int)iTimeout :(int **)iMatchStatus;
- (unsigned int) btBioVerifyLeftNoAsync:(int)iTimeout :(int **)iMatchStatus;
- (unsigned int) ReadFingerPrint :(NSData **) outpk1 :(NSData **) outpk2;
- (unsigned int) GetMyKadFingerPrint;   //Read MyKad fingerprint
- (unsigned int) btBioEnrollExportPK:(int)iExportImage :(uint8_t)templateType :(int)iTimeout :(uint8_t *)oTemplateData :(int *)ioTemplateLength :(uint8_t *)oImageData :(int *)ioImageLength;
- (unsigned int) btBioEnrollExportPKNoAsync:(int)iExportImage :(uint8_t)templateType :(int)iTimeout :(uint8_t *)oTemplateData :(int *)ioTemplateLength :(uint8_t *)oImageData :(int *)ioImageLength;
- (unsigned int) btBioVerifyPK :(uint8_t)templateType :(uint8_t *)TemplateData :(int)inTemplateLen :(int)iTimeout :(int **)iMatchStatus;
- (unsigned int) btBioVerifyPKNoAsync :(uint8_t)templateType :(uint8_t *)TemplateData :(int)inTemplateLen :(int)iTimeout :(int **)iMatchStatus;
- (unsigned int) btBioVerifyPKBothFP :(uint8_t)templateType :(uint8_t *)TemplateData1 :(int)inTemplateLen1 :(uint8_t *)TemplateData2 :(int)inTemplateLen2 :(int)iTimeout :(int **)iMatchStatus;
- (unsigned int) btBioVerifyPKBothFPNoAsync :(uint8_t)templateType :(uint8_t *)TemplateData1 :(int)inTemplateLen1 :(uint8_t *)TemplateData2 :(int)inTemplateLen2 :(int)iTimeout :(int **)iMatchStatus;


// Generic Smartcard functions
- (unsigned int) ConnectSCCard:(uint8_t)slot :(uint8_t *)pResponse :(uint8_t *)pResponseLen;
//- (unsigned int) SCTransmit:(uint8_t *)inCmdBuffer :(uint32_t)inBufferLen :(uint8_t *)outCmdBuffer :(uint32_t *)oBufferLen;
- (unsigned int) SCTransmit:(uint8_t *)inCmdBuffer :(uint8_t)inBufferLen :(uint8_t *)outCmdBuffer :(uint8_t *)oBufferLen;


// MyKad functions
- (unsigned int) initReader;
//- (unsigned int) initReader:(int) slot;
- (unsigned int) ConnectCard;
- (unsigned int) getSlotStatus: (int *) iCardStatus; //1-card present, 0-card absent
- (unsigned int) getCardType; //0-MyKAD, 1- EBA MYKAD, 2- MYKID
- (unsigned int) MyKadOriName:(NSData **) outdata;
- (unsigned int) MyKadGMPCName:(NSData **) outdata;
- (unsigned int) MyKadKPTShortName:(NSData **) outdata;
- (unsigned int) MyKadIDNum:(NSData **) outdata;
- (unsigned int) MyKadOldIDNum:(NSData **) outdata;
- (unsigned int) MyKadAdd1:(NSData **) outdata;
- (unsigned int) MyKadAdd2:(NSData **) outdata;
- (unsigned int) MyKadAdd3:(NSData **) outdata;
- (unsigned int) MyKadPKRight:(NSData **) outdata;
- (unsigned int) MyKadPKLeft:(NSData **) outdata;
- (unsigned int) MyKadGender:(NSData **) outdata;
- (unsigned int) MyKadCitizenship:(NSData **) outdata;
- (unsigned int) MyKadCity:(NSData **) outdata;
- (unsigned int) MyKadBirthPlace:(NSData **) outdata;
- (unsigned int) MyKadBirthDate:(NSData **) outdata;
- (unsigned int) MyKadIssueDate:(NSData **) outdata;
- (unsigned int) MyKadImage:(NSData **) outdata;
- (unsigned int) MyKadState:(NSData **) outdata;
- (unsigned int) MyKadPostcode:(NSData **) outdata;
- (unsigned int) MyKadRace:(NSData **) outdata;
- (unsigned int) MyKadReligion:(NSData **) outdata;
- (unsigned int) MyKadEMOrigin:(NSData **) outdata;
- (unsigned int) MyKadRJ:(NSData **) outdata;
- (unsigned int) MyKadKT:(NSData **) outdata;
- (unsigned int) MyKadOtherID:(NSData **) outdata;
- (unsigned int) MyKadOtherIDCategory:(NSData **) outdata;
- (unsigned int) MyKadGreenCardExp:(NSData **) outdata;
- (unsigned int) MyKadCardVer:(NSData **) outdata;
- (unsigned int) MyKadGreenCardNationality:(NSData **) outdata;
- (unsigned int) getDetails:(struct CardInfo *) CardInfo;
- (unsigned int) DisconnectCard;
- (unsigned int) CloseReader;

// MyKid functions
- (unsigned int) getMyKidDetails:(struct MyKidInfo *) myKidInfo;
- (unsigned int) MyKidGetJPN1Version:(NSData **) outdata;
- (unsigned int) MyKidGetJPN1Size:(NSData **) outdata;
- (unsigned int) MyKidGetBirthCertNo:(NSData **) outdata;
- (unsigned int) MyKidGetIDNo:(NSData **) outdata;
- (unsigned int) MyKidGetName:(NSData **) outdata;
- (unsigned int) MyKidGetGender:(NSData **) outdata;
- (unsigned int) MyKidGetCitizenship:(NSData **) outdata;
- (unsigned int) MyKidGetBirthState:(NSData **) outdata;
- (unsigned int) MyKidGetAddress1:(NSData **) outdata;
- (unsigned int) MyKidGetAddress2:(NSData **) outdata;
- (unsigned int) MyKidGetAddress3:(NSData **) outdata;
- (unsigned int) MyKidGetPostcode:(NSData **) outdata;
- (unsigned int) MyKidGetCity:(NSData **) outdata;
- (unsigned int) MyKidGetState:(NSData **) outdata;
- (unsigned int) MyKidGetFatherReligion:(NSData **) outdata;
- (unsigned int) MyKidGetMotherReligion:(NSData **) outdata;
- (unsigned int) MyKidGetJPN2Version:(NSData **) outdata;
- (unsigned int) MyKidGetJPN2Size:(NSData **) outdata;
- (unsigned int) MyKidGetBirthDate:(NSData **) outdata;
- (unsigned int) MyKidGetBirthTime:(NSData **) outdata;
- (unsigned int) MyKidGetBirthPlace1:(NSData **) outdata;
- (unsigned int) MyKidGetBirthPlace2:(NSData **) outdata;
- (unsigned int) MyKidGetRegistrationDate:(NSData **) outdata;
- (unsigned int) MyKidGetNewName:(NSData **) outdata;
- (unsigned int) MyKidGetNewAdd1:(NSData **) outdata;
- (unsigned int) MyKidGetNewAdd2:(NSData **) outdata;
- (unsigned int) MyKidGetNewAdd3:(NSData **) outdata;
- (unsigned int) MyKidGetNewPostcode:(NSData **) outdata;
- (unsigned int) MyKidGetNewCity:(NSData **) outdata;
- (unsigned int) MyKidGetNewState:(NSData **) outdata;
- (unsigned int) MyKidGetMotherID:(NSData **) outdata;
- (unsigned int) MyKidGetMOtherDocNo:(NSData **) outdata;
- (unsigned int) MyKidGetMOtherDocType:(NSData **) outdata;
- (unsigned int) MyKidGetMotherName:(NSData **) outdata;
- (unsigned int) MyKidGetMotherDOB:(NSData **) outdata;
- (unsigned int) MyKidGetMotherCitizenship:(NSData **) outdata;
- (unsigned int) MyKidGetMotherRace:(NSData **) outdata;
- (unsigned int) MyKidGetFatherID:(NSData **) outdata;
- (unsigned int) MyKidGetFOtherDocNo:(NSData **) outdata;
- (unsigned int) MyKidGetFOtherDocType:(NSData **) outdata;
- (unsigned int) MyKidGetFatherName:(NSData **) outdata;
- (unsigned int) MyKidGetFatherDOB:(NSData **) outdata;
- (unsigned int) MyKidGetFatherCitizenship:(NSData **) outdata;
- (unsigned int) MyKidGetFatherRace:(NSData **) outdata;
- (unsigned int) MyKidGetRegistrationPlace:(NSData **) outdata;
- (unsigned int) MyKidGetJPN4Ver:(NSData **) outdata;
- (unsigned int) MyKidGeJPN4Size:(NSData **) outdata;
- (unsigned int) MyKidGetContactName:(NSData **) outdata;
- (unsigned int) MyKidGetContactAddress:(NSData **) outdata;
- (unsigned int) MyKidGetContactTelNum:(NSData **) outdata;

@end

@protocol BLEReaderDelegate <NSObject>
@required
- (void) VerifyStatusCallBack:(int *)iStatus :(int)iLen :(u_char *)data;

@optional
- (void) pinblockCallBack:(NSString *)data :(int *)UiStatus; //pinblock data, pinpad UI load status (0=dismissed, 1=loaded)
- (void) EnrollStatusCallBack:(int *)iStatus :(int)iLen :(u_char *)data;
- (void) didConnectReader:(int *)iConnectionStatus; // to get the status of the BT device connection
- (void) didBLEPowerOnState:(int *)iPowerOnState; //to get BLE power on state, scanning only after power on state
@end
